#!/usr/bin/ruby

require 'gtkglext'
require 'libglade2'

require 'rb-inotify'

require 'camera'
require 'player'

RESOLUTIONS = [
    [ 1280, 1024 ],
    [ 1152, 864 ],
    [ 1024, 768 ],
    [ 800, 600 ],
    [ 640, 480 ],
    [ 512, 384 ],
    [ 480, 360 ],
    [ 320, 240 ],
    [ 240, 180 ],
    [ 160, 120 ],
    [ 80, 60 ],
    [ 32, 24 ]
]

class UI < Gtk::Builder
 def initialize(cl)
    @cl = cl
    
    super()
    
    add_from_file('pfview.glade')
    connect_signals { |handler| method(handler) }

    @area = get_object('draw')
    @area.set_gl_capability(Gdk::GLConfig.new(Gdk::GLConfig::MODE_DEPTH | Gdk::GLConfig::MODE_DOUBLE | Gdk::GLConfig::MODE_RGB))
    @area.set_size_request(@cl.max_width, @cl.max_height)

    status = get_object('statusbar')

    @empty_pixbuf = Gdk::Pixbuf.new((0..(3*@cl.max_width*@cl.max_height-1)).map { 0.chr }.join, Gdk::Pixbuf::ColorSpace::RGB, false, 8, @cl.max_width, @cl.max_height, @cl.max_width)

    rs = get_object('resolution_store')
    
    @invalid_resolutions = 0
    RESOLUTIONS.each { |i|
	if (i[0] > @cl.max_width) or (i[1] > @cl.max_height) then 
	    @invalid_resolutions += 1
	else 
	    rs.append()[0] = sprintf('%ix%i',i[0],i[1])
	end
    }
    
    resolution = get_object('resolution')
    resolution.active = 0
    resolution.signal_connect('changed') { |widget|
	@cl.set_resolution(RESOLUTIONS[widget.active + @invalid_resolutions])
	draw_expose_event_cb(@area, nil)
    }
    
    exposure = get_object('exposure')
    exposure.value = @cl.exposure
    
    camera_mode = get_object('camera_mode')
    camera_mode.active = (@cl.camera_mode > 0)?1:0

    update_archive_store

    notifier = INotify::Notifier.new
    notifier_io = notifier.to_io

    notifier.watch("saved_camera_images", :moved_from, :moved_to, :create, :delete) {
	update_archive_store
    }

    GLib::Timeout.add(1000) {
	notifier.process if IO.select([notifier_io], [], [], 0)
	true
    }
    

    @pending_text = nil
    
    GLib::Timeout.add(100) {
	if @pending_text
	    status.text = @pending_text
	    @pending_text = nil
	end
	true
    }
    


    get_object('window1').visible = true

    fps_value_changed_cb(get_object('fps'))
    display_mode_changed_cb(get_object('display_mode'))

    @cl.set_procs(
	lambda{ |text, atonce|
	    if atonce 
		#Expecting this called from main thread
		status.text = text 
		Gtk.main_iteration while Gtk.events_pending?
	    else 
	        #This causing crash and blocking as well not working good
		#status.text = text 
		@pending_text = text
	    end
	},
	lambda{ |width, height, data|
	    Gdk::RGB.draw_gray_image(@area.window, @area.style.fg_gc(@area.state), 0, 0, width, height, Gdk::RGB::DITHER_NORMAL, data, width)
	}
    )
 end

 def gtk_widget_delete(user_data, ev)
    kill_clicked_cb(nil)
 end

 def gtk_main_quit(user_data)
    Gtk.main_quit
 end
 
 def gtk_widget_hide(widget)
    widget.visible = false
 end

 def fps_value_changed_cb(widget)
    @cl.display_fps =  widget.value
 end

 def display_mode_changed_cb(widget)
    @cl.display_mode = widget.active
 end

 def exposure_value_changed_cb(widget)
    @cl.exposure = widget.value
 end
 
 def camera_mode_changed_cb(widget)
    @cl.camera_mode = widget.active
 end
 
 def draw_expose_event_cb(area, event)
    @area.window.draw_pixbuf(nil, @empty_pixbuf, 0, 0, 0, 0, @cl.max_width, @cl.max_height, Gdk::RGB::DITHER_NORMAL, 0, 0)
 end


 def capture_clicked_cb(widget)
    widget.sensitive = false
    get_object('stop').sensitive = true
    get_object('replay').sensitive = false

    save = get_object('save')
    storage = @cl.start_capture(save.active, get_object('time').value) {
	get_object('stop').sensitive = false
	get_object('capture').sensitive = true
	get_object('replay').sensitive = true
    }
    
    if (storage > 0) then 
	save.active = storage
    end
 end

 def stop_clicked_cb(widget)
    @cl.stop_capture
 end

 def replay_clicked_cb(widget)
    @cl.replay { |width, height, images, frames|
	Player.new(get_object('fps').value, width, height, images, frames)
    }
 end
 
 
 def update_archive_store()
    a = get_object('archive')
    s = get_object('archive_store')

    cur = a.active
    cur_dir = (a.active >= 0)?s.get_value(a.active_iter, 1):nil

    curid = 0
    active = -1
    
    s.clear
    Dir.foreach("saved_camera_images") { |fn|
	if fn =~ /^(\d{8}_\d{6})(\.(.+))?$/
	    item = s.append()
	    item[0] = $2?sprintf("%s (%s)",$3, $1):"Unknown (#{fn})"
	    item[1] = fn

	    active = curid if cur_dir == fn
	    curid += 1
	end
    }
    
    if curid > 0 then
	a.sensitive = true
	a.active = active
	if active >= 0 then
	    get_object('play').sensitive = true
	    get_object('delete').sensitive = true
	else
	    get_object('play').sensitive = false
	    get_object('delete').sensitive = false
	end	
    else
	a.active = -1
	a.sensitive = false
	get_object('play').sensitive = false
	get_object('delete').sensitive = false
    end
 end
 
 def archive_changed_cb(widget)
    if widget.active >= 0
	get_object('play').sensitive = true 
	get_object('delete').sensitive = true 
    end
 end
 
 def play_clicked_cb(widget)
    active_iter = get_object('archive').active_iter
    active = get_object('archive_store').get_value(active_iter, 1)
    Player.new(get_object('fps').value, nil, nil, active, nil)
 end

 def delete_clicked_cb(widget)
    active_iter = get_object('archive').active_iter
    active = get_object('archive_store').get_value(active_iter, 1)
    system "rm -rf saved_camera_images/#{active}"
 end
 
 def kill_clicked_cb(widget)
    system("killall -9 RTCorrCode_GS &> /dev/null")
 end

 def destroy()
    get_object('window1').destroy
 end
end

Thread.abort_on_exception = true

$freemem = Integer(`free -b | awk 'NR==3 { print $4 }'`)

cl = Camera.new
ui = UI.new(cl)

#It interupts Fg_ function in second thread and exception is comming from there... 
#Kernel.trap("INT") {
#    ui.destroy
#}


#GLib::Idle.add { cl.iteration }

# We need to do it here, otherewise GC.start call in camera will took to long
GC.start

cl.open
Gtk.main
cl.close
