#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <errno.h>
#include <time.h>
#include <stdarg.h>

#include <glib.h>

#include "ds_log.h"

#undef DS_LOG_SUPPORT_THREADS

#define DS_MAX_LOG_STRING 1024
#define DS_MAX_FORMAT_CHARS 32

static int ds_log_time = 0;
static int ds_log_thread = 0;
static int ds_log_level = ((G_LOG_LEVEL_DEBUG<<1) - 1)&G_LOG_LEVEL_MASK;

static int ds_log_exception_message = 1;
static int ds_exception_level = G_LOG_LEVEL_ERROR|G_LOG_LEVEL_CRITICAL;
static DSExceptionHandler ds_exception_handler = NULL;

static char printf_integer_keys[] = "diouxX";
static char printf_accepted_flags[] = "#0- +'I*$123456789.hlLqjzt";

#ifdef DS_LOG_SUPPORT_THREADS
GStaticMutex ds_log_mutex = G_STATIC_MUTEX_INIT;
#endif /* DS_LOG_SUPPORT_THREADS */

/*
static void ds_log_handler_dummy(const gchar *log_domain, GLogLevelFlags log_level, const gchar *message, gpointer user_data) {
}
*/

static void ds_log_handler_stderr(const gchar *log_domain, GLogLevelFlags log_level, const gchar *message, gpointer user_data) {
    fprintf(stderr, "%s\n", message);
}


//static void (*ds_log_handler)(const gchar *log_domain, GLogLevelFlags log_level, const gchar *message, gpointer user_data) = ds_log_handler_stderr;


void ds_log_set_handler(GLogFunc handler, gpointer arg) {
    g_log_set_default_handler(handler?handler:g_log_default_handler, arg);
}

void ds_log_set_level(GLogLevelFlags log_level) {
    ds_log_level = ((log_level<<1) - 1)&G_LOG_LEVEL_MASK;
}

void ds_configure_logger(GLogLevelFlags log_level, const char *sink) {
    ds_log_set_level(log_level);
	// Ignoring at the moment
    ds_log_set_handler(ds_log_handler_stderr, NULL);
}

void ds_log_enable_timestamps() {
    ds_log_time = 1;
}

void ds_log_enable_threads() {
    ds_log_thread = 1;
}

void ds_log_configure_exception_handler(GLogLevelFlags log_level, DSExceptionHandler handler, int log_exception_message) {
    ds_exception_level = log_level;
    ds_exception_handler = handler;
    ds_log_exception_message = log_exception_message;
}

/*
void ds_log_set_domain_level(const gchar *log_domain, GLogLevelFlags log_level) {
    ds_log_level = ((log_level<<1) - 1)&G_LOG_LEVEL_MASK;
    //g_log_set_handler(log_domain,  (~ds_log_level&G_LOG_LEVEL_MASK), &ds_log_handler_dummy);
}

void ds_log_set_domain_handler(const gchar *log_domain, GLogFunc handler, gpointer arg) {
    guint id;

    id = g_log_set_handler(log_domain, G_LOG_LEVEL_MASK, handler, arg);

    // assuming current implementation using sequential uid numbers starting from 1, the handler should not be set directly after that
    if (id > 1) g_log_remove_handler(log_domain, id - 1);
}

int ds_configure_domain_logger(GLogLevelFlags log_level, const char *sink) {
    ds_log_set_level(log_domain, log_level);
	// Ignoring at the moment
    ds_log_set_handler(log_domain, ds_log_handler_stderr, NULL);
}
*/

void _ds_gen_message(int msglen, /* the buffer should be one byte longer for null termination */
	    char *msg,
	    const char *file, 
	    const char *func,
	    int line, 
	    const gchar *service,
	    const gchar *module, 
	    void *ptr, 
	    gchar *prefix,
	    int err, 
	    gchar *format, 
	    va_list ap) {
    struct tm t;
    struct tm gm;
    long offset;
    int hoffset, moffset;
    GTimeVal ts;
    const char *errmsg;
    int flag, pos = 0;
    size_t length, prefixlength;
    gchar *prefixptr, *ppos;

    char subformat[DS_MAX_FORMAT_CHARS + 1] = "";


    if (ds_log_time) {
	g_get_current_time(&ts);
	
	if  ((localtime_r(&ts.tv_sec, &t))&&(!gmtime_r(&ts.tv_sec, &gm))) {
	    offset = (t.tm_sec - gm.tm_sec) + 60*((t.tm_min - gm.tm_min) + 60*(t.tm_hour - gm.tm_hour));
	    if ((t.tm_mday != gm.tm_mday)||(t.tm_mon != gm.tm_mon)||(t.tm_year != gm.tm_year)) {
		if (t.tm_year > gm.tm_year) offset += 86400;
		else if (t.tm_year < gm.tm_year) offset -= 86400;
		else if (t.tm_mon > gm.tm_mon) offset += 86400;
		else if (t.tm_mon < gm.tm_mon) offset -= 86400;
		else if (t.tm_mday > gm.tm_mday) offset += 86400;
		else offset -= 86400;
	    }	
    
	    g_snprintf(msg + pos, msglen - pos + 1, 
		"%04u-%02u-%02uT%02u:%02u:%02u.%07lu",
		t.tm_year+1900,t.tm_mon+1,t.tm_mday,
		t.tm_hour,t.tm_min,t.tm_sec,
		ts.tv_usec*10
	    );
	    
	    pos += strlen(msg + pos);

	    if ((offset==0)&&(pos < msglen)) msg[pos++] = 'Z';
	    else {
		hoffset=offset/3600;
		moffset=abs(offset/60-hoffset*60);
		g_snprintf(msg + pos, msglen - pos + 1, "%+03i:%02u", hoffset, moffset);
	        pos += strlen(msg + pos);
	    }
	} else {
	    g_snprintf(msg + pos, msglen - pos + 1, 
		"%04u-%02u-%02uT%02u:%02u:%02u.%07uZ",
		0,0,0,
		0,0,0,
		0
	    );
	    pos += strlen(msg + pos);
	}
	if (pos < msglen) msg[pos++] = ' ';
    }

    flag = 0;
    if (service) {
	if (ds_log_thread) g_snprintf(msg + pos, msglen - pos + 1, "%s(%p)", service, g_thread_self());
	else g_snprintf(msg + pos, msglen - pos + 1, "%s", service);
	pos += strlen(msg + pos);
    }
    
    if (module) {
	if ((flag)&&(pos < msglen)) msg[pos++] = '/';

	if (ptr) g_snprintf(msg + pos, msglen - pos + 1, "%s(%p): ", module, ptr);
	else g_snprintf(msg + pos, msglen - pos + 1, "%s: ", module);

	pos += strlen(msg + pos);
    } else if ((flag)&&(pos < msglen)) msg[pos++] = '/';
    

    if (err < 0) {
#if (defined(_WIN32)||defined(_WIN64))
        err = GetLastError(); 
#else
        err = errno;
#endif /* WIN */
    }

    if (prefix) {
	prefixptr = prefix;
	for (ppos = strchr(prefix, '%');ppos;) {
	    length = strspn(ppos + 1, printf_accepted_flags) + 1;
	    if ((length < DS_MAX_FORMAT_CHARS)&&((ppos[length]=='s')||(strspn(ppos+length, printf_integer_keys)>0))) {
		prefixlength = (ppos - prefixptr);
		if (prefixlength) {
		    if ((msglen - pos) > prefixlength) g_strlcpy(msg + pos, prefixptr, prefixlength + 1);
		    else g_strlcpy(msg + pos, prefixptr, msglen - pos + 1);
		    pos += strlen(msg + pos);
		}

		if (ppos[length]=='s') g_vsnprintf(msg + pos, msglen - pos + 1, format, ap);
		else {
		    strncpy(subformat, ppos, length + 1); subformat[length + 1] = 0;
		    g_snprintf(msg + pos, msglen - pos + 1, subformat, err);
		}
		pos += strlen(msg + pos);
		    
		prefixptr = ppos + length + 1;
		ppos = strchr(prefixptr, '%');
	    } else if (ppos[length]) {
		ppos = strchr(ppos + length + 1, '%');
	    } else {
		break;
	    }
	}
	
	if (*prefixptr) {
	    g_strlcpy(msg + pos, prefixptr, msglen - pos + 1);
	    pos += strlen(msg + pos);
	}
    } else {	
	if (err) {
#if (defined(_WIN32)||defined(_WIN64))
	    errmsg = NULL;
#else
    	    errmsg = g_strerror(err);
#endif /* WIN */

	    if (errmsg) g_snprintf(msg + pos, msglen - pos + 1, "Error %i(%s): ", err, errmsg);
	    else g_snprintf(msg + pos, msglen - pos + 1, "Error %i: ", err);

	    pos += strlen(msg + pos);
	}

	if (format) {
	    g_vsnprintf(msg + pos, msglen - pos + 1, format, ap);

	    pos += strlen(msg + pos);
	}
    }
    
    if (file) {
	if (func) g_snprintf(msg + pos, msglen - pos + 1, " <%s:%s:%u>", file, func, line);
	else g_snprintf(msg + pos, msglen - pos + 1, " <%s:%u>", file, line);
	pos += strlen(msg + pos);
    }
    
//    if (pos < msglen) msg[pos++] = '\n';
    msg[pos] = 0;
}

int _ds_log_message(int condition, 
	    const gchar *log_domain, 
	    GLogLevelFlags log_levels, 
	    const char *file, 
	    const char *func,
	    int line, 
	    const gchar *service,
	    const gchar *module, 
	    void *ptr, 
	    gchar *prefix,
	    int err, 
	    gchar *format, 
	    ...) {

    va_list ap;
    char msg[DS_MAX_LOG_STRING + 1] = "";
    
    if (condition) return 0;
    if ((ds_log_level&log_levels)==0) return 1;

    va_start(ap, format);
    _ds_gen_message(DS_MAX_LOG_STRING, msg, file, func, line, service, module, ptr, prefix, err, format, ap);
    va_end(ap);

    if (log_levels&DS_LOG_CLEAN_MESSAGE) {
	if (prefix) g_free(prefix);
	else if (format) g_free(format);
    }

    if ((!ds_exception_handler)||(log_levels&ds_exception_level==0)||(ds_log_exception_message)) {
#ifdef DS_LOG_SUPPORT_THREADS
	g_static_mutex_lock(g_log_mutex);
#endif /* DS_LOG_SUPPORT_THREADS */
	g_log(log_domain, log_levels&DS_LOG_LEVEL_MASK, msg);
#ifdef DS_LOG_SUPPORT_THREADS
	g_static_mutex_unlock(g_log_mutex);
#endif /* DS_LOG_SUPPORT_THREADS */
    }

    if ((ds_exception_handler)&&(log_levels&ds_exception_level)) {
	ds_exception_handler(err, msg);
    } 
    
    if (log_levels&DS_LOG_FATAL_MESSAGE) {
	if (file) g_error(" Program stops at %s:%u. Bye\n", file, line);
	else g_error(" Program stops. Bye\n");
	exit(1);
    }
    
    return 1;
}


