#include <stdio.h>
#include <stdlib.h>
#include <string.h>

#define fg_apc_data TCameraLink

#include <fgrab_struct.h>
#include <fgrab_prototyp.h>
#include <fgrab_define.h>

#include "ds_log.h"
#include "cl.h"

#define DLLNAME 		"libFullAreaGray8.so"
#define CL_BOARD 		0
#define CL_PORT			PORT_A			// Full speed, PORT_AB ?
#define CL_SAMPLES_PER_PIXEL 	1
#define CL_BYTES_PER_SAMPLE 	1
#define CL_BUFFERS		16
#define CL_SECURITY_BUFFERS	2			// We will skip the buffer if less than that amount of free buffers remained before overwrite
#define CL_TIMEOUT		2			// In seconds


static int cl_new_frame_callback(frameindex_t frame, CameraLink *ctx) {
    void *data = Fg_getImagePtr(ctx->fg, frame, CL_PORT);
    ctx->cb(ctx->cbctx, frame,  data);
    return 0;
}



CameraLink *cl_create() {
    CameraLink *ctx = malloc(sizeof(CameraLink));
    if (ctx) {
	memset(ctx, 0, sizeof(CameraLink));
    }
}

int cl_init(CameraLink *ctx) {
    return 0;
}

void cl_free(CameraLink *ctx) {
    g_assert(ctx);
    
    cl_close(ctx);
}

void cl_destroy(CameraLink *ctx) {
    cl_free(ctx);    
    free(ctx);
}

int cl_set_max_resolution(CameraLink *ctx, int width, int height) {
    g_assert(ctx);
    g_assert(!ctx->running);

    ctx->max_width = width;
    ctx->max_height = height;
}

int cl_set_resolution(CameraLink *ctx, int width, int height) {
    int err;
    
    g_assert(ctx);
    g_assert(ctx->running);
    g_assert(width <= ctx->max_width);
    g_assert(height <= ctx->max_height);    

    err = Fg_setParameter(ctx->fg, FG_WIDTH, &width, CL_PORT);
    ds_exception_reterr(err == 0, -1, "Fg_setParameter(width = %i) failed: %s", width, Fg_getLastErrorDescription(ctx->fg));

    err = Fg_setParameter(ctx->fg, FG_HEIGHT, &height, CL_PORT);
    ds_exception_reterr(err == 0, -1, "Fg_setParameter(height = %i) failed: %s", height, Fg_getLastErrorDescription(ctx->fg));

    ctx->width = width;
    ctx->height = height;

    return 0;
}

int cl_open(CameraLink *ctx, int width, int height, int flags) {
    int err;
    size_t buf_size;
    void *buf;
    
    double frame_rate;
    
    g_assert(ctx);
    
    if (!ctx->max_width) ctx->max_width = width;
    if (!ctx->max_height) ctx->max_height = height;
    
    if (ctx->fg) cl_close(ctx);
    
    ctx->fg = Fg_Init(DLLNAME, CL_BOARD);
    ds_exception_reterr(ctx->fg != NULL, -1, "Error in Fg_Init: %s", Fg_getLastErrorDescription(NULL));
    
    buf_size = ctx->max_width * ctx->max_height * CL_SAMPLES_PER_PIXEL * CL_BYTES_PER_SAMPLE * CL_BUFFERS;
    
    buf = Fg_AllocMem(ctx->fg, buf_size, CL_BUFFERS, CL_PORT);
    ds_exception_reterr(buf != NULL, -1, "Error in Fg_AllocMem (%i bytes): %s", buf_size, Fg_getLastErrorDescription(NULL));

    err = Fg_setParameter(ctx->fg, FG_WIDTH, &width, CL_PORT);
    ds_exception_reterr(err == 0, -1, "Fg_setParameter(width = %i) failed: %s", width, Fg_getLastErrorDescription(ctx->fg));

    err = Fg_setParameter(ctx->fg, FG_HEIGHT, &height, CL_PORT);
    ds_exception_reterr(err == 0, -1, "Fg_setParameter(height = %i) failed: %s", height, Fg_getLastErrorDescription(ctx->fg));
    
    if (ctx->cb) {
        struct FgApcControl ctrl;

	ctrl.version = 0;
	ctrl.data = (struct fg_apc_data*)ctx;
	ctrl.func = &cl_new_frame_callback;
	ctrl.flags = FG_APC_DEFAULTS;
	ctrl.timeout = 1;
    
	err = Fg_registerApcHandler(ctx->fg, CL_PORT, &ctrl, FG_APC_CONTROL_BASIC); 
	ds_exception_reterr(err == 0, -1, "Fg_registerApcHandler() failed: %s", Fg_getLastErrorDescription(ctx->fg));
    }

    if (flags&CL_FLAGS_SECURE) {
	err = Fg_AcquireEx(ctx->fg, CL_PORT, GRAB_INFINITE, ACQ_BLOCK, buf);
    } else {
	err = Fg_Acquire(ctx->fg, CL_PORT, GRAB_INFINITE);
    }
    
    ds_exception_reterr(err == 0, -1, "Fg_Acquire() failed: %s", Fg_getLastErrorDescription(ctx->fg));

    ctx->running = 1;
    ctx->width = width;
    ctx->height = height;
    
    return 0;
}

int cl_close(CameraLink *ctx) {
    ctx->running = 0;
    
    if (ctx->fg) {
	Fg_stopAcquire(ctx->fg, CL_PORT);
	if (ctx->cb) Fg_registerApcHandler(ctx->fg, CL_PORT, NULL, FG_APC_CONTROL_BASIC);
	Fg_FreeMem(ctx->fg, CL_PORT);
	Fg_FreeGrabber(ctx->fg);
	ctx->fg = NULL;
    }
    
    return 0;
}


int cl_register_frame_callback(CameraLink *ctx, TCameraFrameCallback callback, void *data) {
    g_assert(ctx);
    gl_assert(!ctx->running);
    
    ctx->cb = callback; 
    ctx->cbctx = data;
    
    return 0;
}


int cl_run_frames_continuous(CameraLink *ctx, int *run_flag, TCameraFrameCallback callback, void *cbctx) {
    frameindex_t cur_frame = 0, last_frame = 1, new_frame;
    void *data;

    g_assert(ctx);
    g_assert(run_flag);
    g_assert(callback);
    g_assert(ctx->running);

    while (*run_flag) {
        new_frame = Fg_getLastPicNumberBlocking(ctx->fg, cur_frame + 1, CL_PORT, CL_TIMEOUT);
	if (new_frame < 0) {
	    continue;
/*	    ds_error("Fg_getLastPicNumberBlocking() failed: %s", Fg_getLastErrorDescription(ctx->fg));
	    return -1;*/
	} else {
	    last_frame = new_frame;
	}
	
	if ((cur_frame + CL_BUFFERS - CL_SECURITY_BUFFERS) < last_frame) {
	    cur_frame = last_frame - CL_BUFFERS + CL_SECURITY_BUFFERS;
	}
	
	data = Fg_getImagePtr(ctx->fg, ++cur_frame, CL_PORT);
	if (data) callback(cbctx, cur_frame, data);
    }

    return 0;
}




//int cl_get_next_frame






/*
int cl_advance_next_frame(CameraLink *ctx) {
    frameindex_t res;
    
    ds_exception_reterr(ctx->running, -1, "CameraLink context is not initialized");

    if (ctx->current_frame == ctx->last_frame) {
	res = Fg_getLastPicNumberBlocking(ctx->fg, ctx->last_frame + 1, CL_PORT, CL_TIMEOUT);
	if (!res) {
	    ds_error("Fg_getLastPicNumberBlocking() failed: %s", Fg_getLastErrorDescription(ctx->fg));
	    return -1;
	}
	
	ctx->last_frame = res;
    }
    
    return 0;
}

int cl_get_frame(CameraLink *ctx) {
    ds_exception_reterr(ctx->running, -1, "CameraLink context is not initialized");

    Fg_getImagePtr(ctx->fg, ctx->current_frame++, CL_PORT);
}
*/
