#include <stdio.h>
#include "ruby.h"

#include "ds_log.h"
#include "cl.h"

#define BUFFER_FRAMES 16

static VALUE module_object;
static VALUE class_object;

struct TCameraLinkReader {
    CameraLink cl;

    VALUE str;
    
    int run_flag;
    int width, height;
    
    pid_t pid;
    int ready;
    int frame_id;
    void *data;
};

typedef struct TCameraLinkReader CameraLinkReader;

static void reader_mark(CameraLink *ctx) {
}

static VALUE reader_allocate(VALUE klass) {
    CameraLink *ctx = malloc(sizeof(CameraLinkReader));
    if (ctx) {
	memset(ctx, 0, sizeof(CameraLinkReader));
    } else {
        rb_raise(rb_eRuntimeError, "Allocation of CameraLink context is failed");
    }
    
    return Data_Wrap_Struct(klass, reader_mark, cl_destroy, ctx);
}


/*
static reader_callback(void *callback, int frame_id, void *data) {
    rb_funcall((VALUE)callback, rb_intern("call"), 2, INT2NUM(frame_id), Qnil);
}
*/

static reader_safe_callback(CameraLink *ctx, int frame_id, void *data) {
    rb_yield_values(2, INT2NUM(frame_id), rb_str_new(data, ctx->width * ctx->height));
}

static reader_callback(CameraLink *ctx, int frame_id, void *data) {
    struct RString *str = (struct RString*)((CameraLinkReader*)ctx)->str;

    char *ptr = str->ptr;
    str->ptr = data;
    rb_yield_values(2, INT2NUM(frame_id), (VALUE)str);
    str->ptr = ptr;

/*
	// This is a bit safer
    memcpy(str->ptr, data, ctx->width * ctx->height);
    rb_yield_values(2, INT2NUM(frame_id), str);
*/

    return 0;
}

static reader_thread_callback(CameraLinkReader *ctx, int frame_id, void *data) {
    while ((ctx->ready)&&(ctx->run_flag)) usleep(100);
    
    ctx->frame_id = frame_id;
    ctx->data = data;
    ctx->ready = 1;
}


static VALUE reader_init(VALUE self) {
    int err;
    CameraLink *ctx;

    Data_Get_Struct (self, CameraLink, ctx);

    err = cl_init(ctx);
    if (err) rb_raise(rb_eRuntimeError, "Initialization of CameraLink context is failed");

//    cl_register_frame_callback(ctx, &reader_callback, (void*)callback);

    return Qnil;
}

static VALUE reader_open(VALUE self, VALUE vWidth, VALUE vHeight) {
    CameraLinkReader *ctx;
    int width = NUM2INT(vWidth);
    int height = NUM2INT(vHeight);
    char tmp[width*height];

    Data_Get_Struct (self, CameraLinkReader, ctx);

    cl_open((CameraLink*)ctx, width, height, 0);
    ctx->str = rb_str_new(tmp, width * height);

    return Qnil;
}

static VALUE reader_set_max_resolution(VALUE self, VALUE vWidth, VALUE vHeight) {
    CameraLink *ctx;

    Data_Get_Struct (self, CameraLink, ctx);
    cl_set_max_resolution((CameraLink*)ctx, NUM2INT(vWidth), NUM2INT(vHeight));

    return Qnil;
}

static VALUE reader_set_resolution(VALUE self, VALUE vWidth, VALUE vHeight) {
    CameraLink *ctx;

    Data_Get_Struct (self, CameraLink, ctx);
    cl_set_resolution((CameraLink*)ctx, NUM2INT(vWidth), NUM2INT(vHeight));

    return Qnil;
}

static VALUE reader_close(VALUE self) {
    CameraLink *ctx;

    Data_Get_Struct (self, CameraLink, ctx);
    cl_close(ctx);

    return Qnil;
}

static int *reader_run_thread(CameraLinkReader *ctx) {
    int err;
    err = cl_run_frames_continuous((CameraLink*)ctx, &ctx->run_flag, reader_thread_callback, ctx);
    return NULL;
}

static VALUE reader_sleep(VALUE delay) {
    return rb_funcall(Qnil, rb_intern("sleep"), 1, delay);
}

static VALUE reader_threaded_run(VALUE self, VALUE run_flag) {
    int err;
    void *res;
    VALUE delay;
    pthread_t thr;

    CameraLinkReader *ctx;

    rb_need_block();
    
    Data_Get_Struct (self, CameraLinkReader, ctx);

    ctx->pid = getpid();
    ctx->run_flag = 1;

    err = pthread_create(&thr, NULL, (void*(*)(void*))reader_run_thread,(void*)ctx);
    if (err) rb_raise(rb_eRuntimeError, "Thread scheduling is failed");
    
    delay = rb_float_new(0.001);	// better implement over socket, signal and exception are not really working

    while (ctx->run_flag) {
	if (ctx->ready) {
	    //printf("In: %i %p\n", ctx->frame_id, ctx->data);
	    reader_callback((CameraLink*)ctx, ctx->frame_id, ctx->data);
	    ctx->ready = 0;
	}
	rb_funcall(Qnil, rb_intern("sleep"), 1, delay);
    }
    
    pthread_join(thr, &res);

    return Qnil;
}

static VALUE reader_run(VALUE self, VALUE run_flag) {
    int err;

    CameraLinkReader *ctx;

    rb_need_block();
    
    Data_Get_Struct (self, CameraLinkReader, ctx);
    
    ctx->run_flag = 1;

    err = cl_run_frames_continuous((CameraLink*)ctx, &ctx->run_flag, reader_callback, ctx);
    if (err) rb_raise(rb_eRuntimeError, "Run is failed");

    return Qnil;
}

static VALUE reader_safe_run(VALUE self, VALUE run_flag) {
    int err;

    CameraLinkReader *ctx;

    rb_need_block();
    
    Data_Get_Struct (self, CameraLinkReader, ctx);

    ctx->run_flag = 1;

    err = cl_run_frames_continuous((CameraLink*)ctx, &ctx->run_flag, reader_safe_callback, ctx);
    if (err) rb_raise(rb_eRuntimeError, "Run is failed");

    return Qnil;
}



static VALUE reader_stop(VALUE self, VALUE run_flag) {
    int err;
    int rflag = 0;

    CameraLinkReader *ctx;

    Data_Get_Struct (self, CameraLinkReader, ctx);

    ctx->run_flag = 0;
    
    return Qnil;
}


void ds_raise_ruby_exception(int err, const char *msg) {
    rb_raise(rb_eRuntimeError, msg);
}

void Init_cameralink() {

    ds_log_configure_exception_handler(G_LOG_LEVEL_CRITICAL, ds_raise_ruby_exception, 0);

    module_object = rb_define_module("CameraLink");
    class_object = rb_define_class_under(module_object, "Reader", rb_cObject);
    rb_define_alloc_func(class_object, reader_allocate);
    rb_define_method(class_object, "initialize", reader_init, 0);
    rb_define_method(class_object, "open", reader_open, 2);
    rb_define_method(class_object, "close", reader_close, 0);
    rb_define_method(class_object, "run", reader_run, 1);
    rb_define_method(class_object, "safe_run", reader_safe_run, 1);
    rb_define_method(class_object, "threaded_run", reader_threaded_run, 1);
    rb_define_method(class_object, "stop", reader_stop, 0);
    rb_define_method(class_object, "set_max_resolution", reader_set_max_resolution, 2);
    rb_define_method(class_object, "set_resolution", reader_set_resolution, 2);
}
